

class Unit {
  constructor(id, name, x, y, hp, atk, def, mov, team) {
    this.id = id;
    this.name = name;
    this.x = x;
    this.y = y;
    this.maxHp = hp;
    this.hp = hp;
    this.atk = atk;
    this.def = def;
    this.mov = mov;
    this.team = team; // 'ally' or 'enemy'
    this.acted = false;
  }

  alive() {
    return this.hp > 0;
  }

  takeDamage(dmg) {
    this.hp -= dmg;
    if (this.hp < 0) this.hp = 0;
  }
}


class BattleMap {
  constructor(width, height) {
    this.width = width;
    this.height = height;
  }

  inBounds(x, y) {
    return x >= 0 && y >= 0 && x < this.width && y < this.height;
  }
}



class TacticsBattle {
  constructor(map, allies, enemies) {
    this.map = map;
    this.allies = allies;
    this.enemies = enemies;
    this.turnTeam = 'ally'; // 'ally' or 'enemy'
    this.selectedUnit = null;
  }

  units(team) {
    if (team === 'ally') return this.allies;
    if (team === 'enemy') return this.enemies;
    return [...this.allies, ...this.enemies];
  }

  livingUnits(team) {
    return this.units(team).filter(u => u.alive());
  }

  currentTeamAllActed() {
    return this.livingUnits(this.turnTeam).every(u => u.acted);
  }

  endTurnIfNeeded() {
    if (this.currentTeamAllActed()) {
      this.units(this.turnTeam).forEach(u => u.acted = false);
      this.turnTeam = this.turnTeam === 'ally' ? 'enemy' : 'ally';
      return this.turnTeam; // ← ここ大事
    }
    return null;
  }

  isBattleFinished() {
    const alliesAlive = this.livingUnits('ally').length > 0;
    const enemiesAlive = this.livingUnits('enemy').length > 0;
    if (!alliesAlive || !enemiesAlive) {
      return true;
    }
    return false;
  }

  damage(user, target) {
    const raw = user.atk - target.def;
    return Math.max(1, raw);
  }

  attack(user, target) {
    const dmg = this.damage(user, target);
    target.takeDamage(dmg);
    user.acted = true;
    this.endTurnIfNeeded();
     return dmg; // ★ ここでダメージ値を返す
  }
}


function Scene_Tactics() {
  this.initialize(...arguments);
}

Scene_Tactics.prototype = Object.create(Scene_Base.prototype);
Scene_Tactics.prototype.constructor = Scene_Tactics;
Scene_Tactics.prototype.enemyStepTowards = function(enemy, target) {
    const candidates = [];

    // X方向
    if (target.x > enemy.x) {
        candidates.push({ x: enemy.x + 1, y: enemy.y });
    } else if (target.x < enemy.x) {
        candidates.push({ x: enemy.x - 1, y: enemy.y });
    }

    // Y方向
    if (target.y > enemy.y) {
        candidates.push({ x: enemy.x, y: enemy.y + 1 });
    } else if (target.y < enemy.y) {
        candidates.push({ x: enemy.x, y: enemy.y - 1 });
    }

    // 候補の中から、マップ内＆空きマスのやつを一つ選ぶ
    for (const c of candidates) {
        if (!this._battle.map.inBounds(c.x, c.y)) continue;
        if (!this.isTileEmpty(c.x, c.y)) continue;
        return c;
    }

    // 動けるマスがなければ何もしない
    return null;
};


Scene_Tactics.prototype.startPhaseBanner = function(team) {
    // 既に出てたら消す
    if (this._phaseSprite) {
        this.removeChild(this._phaseSprite);
        this._phaseSprite = null;
    }

    const w = 260;
    const h = 60;
    const bmp = new Bitmap(w, h);

    // 背景
    bmp.fillRect(0, 0, w, h, "rgba(0, 0, 0, 0.7)");

    // 枠線
    bmp.fillRect(0, 0, w, 2, "#ffffff");
    bmp.fillRect(0, h - 2, w, 2, "#ffffff");
    bmp.fillRect(0, 0, 2, h, "#ffffff");
    bmp.fillRect(w - 2, 0, 2, h, "#ffffff");

    // テキスト
    bmp.textColor = "#ffffff";
    bmp.outlineColor = "#000000";
    bmp.outlineWidth = 4;

    const text = (team === "ally") ? "味方ターン" : "敵ターン";
    bmp.drawText(text, 0, 10, w, 40, "center");

    const sprite = new Sprite(bmp);
    sprite.x = (Graphics.boxWidth - w) / 2;
    sprite.y = (Graphics.boxHeight - h) / 2;
    sprite.opacity = 0;   // 最初は透明

    this._phaseSprite = sprite;
    this._phaseTimer = 60; // 60フレームくらい表示（1秒前後）

    this.addChild(sprite);
};


Scene_Tactics.prototype.showDamageText = function(target, dmg) {
    const cellW = this._cellW;
    const cellH = this._cellH;

    // ダメージ表示用ビットマップ
    const bmp = new Bitmap(48, 24);
    bmp.fontSize = 20;
    bmp.textColor = "#ffffff";
    bmp.outlineColor = "#000000";
    bmp.outlineWidth = 4;
    bmp.drawText(String(dmg), 0, 0, bmp.width, bmp.height, "center");

    const sprite = new Sprite(bmp);

    // ターゲットマスの中央あたりに配置
    sprite.x = target.x * cellW + (cellW - bmp.width) / 2;
    sprite.y = target.y * cellH + (cellH - bmp.height) / 2;

    // アニメ用の簡単なプロパティ
    sprite._vy = -1;   // 上方向の速度
    sprite._life = 30; // 生存フレーム数

    this._damageTexts.push(sprite);
    this.addChild(sprite);
};









Scene_Tactics.prototype.tryActSelectedUnit = function() {
    const u = this._selectedUnit;
    if (!u) return;

    const tx = this._cursorX;
    const ty = this._cursorY;
    const target = this.unitAt(tx, ty);

    const dist = Math.abs(tx - u.x) + Math.abs(ty - u.y);

    // ★ 1. カーソル先に敵がいて、隣接マスなら「攻撃」
    if (target && target.team !== u.team && dist === 1) {
            const dmg = this._battle.attack(u, target); // ★ダメージ値を取得
    this.showDamageText(target, dmg);          // ★数字を出す
        this._selectedUnit = null;
        this._moveRange = []; // ★
        this.redrawBoard();
const newTeam = this._battle.endTurnIfNeeded();
if (newTeam) {
    this.startPhaseBanner(newTeam);
}
        // 戦闘終了ならシーンを閉じる
        if (this._battle.isBattleFinished()) {
            SceneManager.pop();
        }
        return;
    }

    // ★ 2. それ以外 → いつもの移動処理へ
    this.tryMoveSelectedUnit();
};





Scene_Tactics.prototype.updateEnemyTurn = function() {
    const enemies = this._battle.livingUnits('enemy');
    const allies  = this._battle.livingUnits('ally');
    if (enemies.length === 0 || allies.length === 0) return;

    for (const enemy of enemies) {
        if (!enemy.alive()) continue;
        if (allies.length === 0) break;

        // 移動可能回数ぶんループ
        for (let stepCount = 0; stepCount < enemy.mov; stepCount++) {

            // 毎ステップごとに一番近い味方を再計算
            let target = null;
            let minDist = Infinity;
            for (const a of allies) {
                const d = Math.abs(a.x - enemy.x) + Math.abs(a.y - enemy.y);
                if (d < minDist) {
                    minDist = d;
                    target = a;
                }
            }
            if (!target) break;

            // 隣接したら攻撃して終了
            if (minDist === 1) {
                const dmg = this._battle.attack(enemy, target);
                this.showDamageText(target, dmg);
                break;
            }

            // まだ遠いなら1マスだけ近づく
            const step = this.enemyStepTowards(enemy, target);
            if (!step) break; // 動けないなら終了

            enemy.x = step.x;
            enemy.y = step.y;
        }

        enemy.acted = true;
    }

  

    // 敵全員行動終わり → 行動フラグリセットして味方ターンへ
    this._battle.units('enemy').forEach(u => u.acted = false);
    this._battle.turnTeam = 'ally';

    this.redrawBoard();
    this.startPhaseBanner('ally');
};


Scene_Tactics.prototype.createMapBackground = function() {
    // Spriteset_Map を丸ごと乗せると、
    // タイル・遠景・イベントグラフィックまで全部描画される
    this._spriteset = new Spriteset_Map();
    this.addChild(this._spriteset);
};



Scene_Tactics.prototype.initialize = function() {
  Scene_Base.prototype.initialize.call(this);
};

Scene_Tactics.prototype.create = function() {
  Scene_Base.prototype.create.call(this);
    this.createWindowLayer();

    // ★ 今いるマップの見た目をそのまま背景に使う
    this.createMapBackground();
  // バトル用データ作成
  const map = new BattleMap(8, 6);
  //id, name, x, y, hp, atk, def, mov, team) {
  const a1 = new Unit(1, "味方1", 1, 1, 20, 8, 3, 3, 'ally');
  const a2 = new Unit(2, "味方2", 1, 3, 18, 7, 2, 3, 'ally');
  const e1 = new Unit(3, "敵1", 6, 1, 30, 9, 2, 1, 'enemy');
  const e2 = new Unit(4, "敵2", 6, 4, 15, 6, 2, 5, 'enemy');

  this._battle = new TacticsBattle(map, [a1, a2], [e1, e2]);
  // ===== セルサイズ計算 =====
  this._cellW = Math.floor(Graphics.boxWidth / this._battle.map.width);
  this._cellH = Math.floor(Graphics.boxHeight / this._battle.map.height);

  // とりあえずテスト用：グリッドを描くスプライト
  this.createBoardSprite();
  
  // ===== カーソル関連 =====
  this._cursorX = 0;
  this._cursorY = 0;
  this._selectedUnit = null; // 選択中ユニット（いなければnull）
  
  this._moveRange = []; // ★移動範囲ハイライト用
  this._damageTexts = [];


    // ★ ターン演出用
    this._phaseSprite = null; // 現在表示中のバナー
    this._phaseTimer = 0;     // 残りフレーム



  this.createCursorSprite();
  this.updateCursorPosition();




Scene_Tactics.prototype.create = function() {
    Scene_Base.prototype.create.call(this);

    // バトル用データ作成とか cellW/cellH 計算はそのまま

    this.createBoardSprite();

    this._cursorX = 0;
    this._cursorY = 0;
    this._selectedUnit = null;
    this._moveRange = [];

    // ★ダメージ数字用リスト
    this._damageTexts = [];

    this.createCursorSprite();
    this.updateCursorPosition();
};







Scene_Tactics.prototype.unitAt = function(x, y) {
  const all = [...this._battle.allies, ...this._battle.enemies];
  return all.find(u => u.alive() && u.x === x && u.y === y) || null;
};

Scene_Tactics.prototype.isTileEmpty = function(x, y) {
  return !this.unitAt(x, y);
};



};

Scene_Tactics.prototype.createBoardSprite = function() {
  const w = Graphics.boxWidth;
  const h = Graphics.boxHeight;
  const bmp = new Bitmap(w, h);
  this._boardSprite = new Sprite(bmp);
  this.addChild(this._boardSprite);
  this.redrawBoard();
};

Scene_Tactics.prototype.redrawBoard = function() {
  const bmp = this._boardSprite.bitmap;
  bmp.clear();
  const cellW = this._cellW;
  const cellH = this._cellH;

  // グリッド線
  for (let y = 0; y < this._battle.map.height; y++) {
    for (let x = 0; x < this._battle.map.width; x++) {
          
      bmp.fillRect(x * cellW, y * cellH, cellW, 1, "#666666");
      bmp.fillRect(x * cellW, y * cellH, 1, cellH, "#666666");
    }
  }


  // ★ここ：_moveRange が undefined/null でも平気なようにする
  const range = this._moveRange || [];
  for (const tile of range) {
    const x = tile.x;
    const y = tile.y;
    bmp.fillRect(
      x * cellW + 2,
      y * cellH + 2,
      cellW - 4,
      cellH - 4,
      "rgba(0, 0, 255, 0.3)"
    );
  }
  // ユニット＋HPバー描画
  this.drawUnitsWithHp(this._battle.allies, "#3344ff");
  this.drawUnitsWithHp(this._battle.enemies, "#ff3333");


  // ユニット描画（味方=青、敵=赤）
  const drawUnits = (units, color) => {
    for (const u of units) {
      if (!u.alive()) continue;
      const cx = u.x * cellW;
      const cy = u.y * cellH;
      bmp.fillRect(cx + 4, cy + 4, cellW - 8, cellH - 8, color);
      bmp.drawText(u.name, cx, cy, cellW, cellH, 'center');
    }
  };
  drawUnits(this._battle.allies, "#3344ff");
  drawUnits(this._battle.enemies, "#ff3333");
};

Scene_Tactics.prototype.drawUnitsWithHp = function(units, color) {
  const bmp = this._boardSprite.bitmap;
  const cellW = this._cellW;
  const cellH = this._cellH;

  for (const u of units) {
    if (!u.alive()) continue;

    const cx = u.x * cellW;
    const cy = u.y * cellH;

    // 本体の四角
    bmp.fillRect(cx + 4, cy + 4, cellW - 8, cellH - 16, color); // 少し上に縮める

    // 名前
    bmp.drawText(u.name, cx, cy + 4, cellW, 24, 'center');

    // ★ HPバー
    const barWidth = cellW - 8;
    const barHeight = 6;
    const barX = cx + 4;
    const barY = cy + cellH - barHeight - 4;

    // 背景（黒）
    bmp.fillRect(barX, barY, barWidth, barHeight, "#000000");

    const rate = u.hp / u.maxHp;
    const hpWidth = Math.max(0, Math.floor(barWidth * rate));

    // 残りHP（緑）
    bmp.fillRect(barX, barY, hpWidth, barHeight, "#00ff00");
  }
};

Scene_Tactics.prototype.updatePhaseBanner = function() {
    if (!this._phaseSprite) return;
    if (this._phaseTimer <= 0) {
        // 念のためクリーンアップ
        this.removeChild(this._phaseSprite);
        this._phaseSprite = null;
        return;
    }

    this._phaseTimer--;

    // 前半でフェードイン、後半でフェードアウト
    const total = 60;
    const half = total / 2;
    const s = this._phaseSprite;

    if (this._phaseTimer > half) {
        // フェードイン
        const t = total - this._phaseTimer; // 0→half
        s.opacity = 255 * (t / half);
    } else {
        // フェードアウト
        const t = this._phaseTimer; // half→0
        s.opacity = 255 * (t / half);
    }
};
Scene_Tactics.prototype.update = function() {
  Scene_Base.prototype.update.call(this);
    // ★ まずバナーを更新
    this.updatePhaseBanner();

    // ★ 演出中はそれ以外止める
    if (this._phaseTimer > 0) {
        // ダメージ数字だけは動かしてOKにする
        this.updateDamageTexts();
        this.refreshStatusWindow();
        return;
    }



    if (this._battle.turnTeam === 'ally') {
        // 味方ターン：今までの操作
        this.updateCursorControl();
        this.updateOkCancel();
    } else {
        // 敵ターン：自動行動
        this.updateEnemyTurn();
    }
    // ★ダメージ演出更新
    this.updateDamageTexts();
    // ★ ステータス更新
    this.refreshStatusWindow();
    if (this._battle.isBattleFinished()) {
        SceneManager.pop();
    }

  // ここに「カーソル」「移動入力」「攻撃入力」「ターン終了処理」を少しずつ足していく
};
Scene_Tactics.prototype.updateDamageTexts = function() {
    for (let i = this._damageTexts.length - 1; i >= 0; i--) {
        const s = this._damageTexts[i];
        s.y += s._vy;
        s._life--;

        if (s._life < 10) {
            s.opacity = 255 * (s._life / 10);
        }

        if (s._life <= 0) {
            this.removeChild(s);
            this._damageTexts.splice(i, 1);
        }
    }
};

// --- ★4-3. カーソル移動（ここに書く） ---
Scene_Tactics.prototype.updateCursorControl = function() {
  let moved = false;

  if (Input.isTriggered("left")) {
    if (this._cursorX > 0) {
      this._cursorX--;
      moved = true;
    }
  } else if (Input.isTriggered("right")) {
    if (this._cursorX < this._battle.map.width - 1) {
      this._cursorX++;
      moved = true;
    }
  } else if (Input.isTriggered("up")) {
    if (this._cursorY > 0) {
      this._cursorY--;
      moved = true;
    }
  } else if (Input.isTriggered("down")) {
    if (this._cursorY < this._battle.map.height - 1) {
      this._cursorY++;
      moved = true;
    }
  }

  // --- 決定／キャンセル処理 ---
Scene_Tactics.prototype.updateOkCancel = function() {
// 単体待機：選択中＋Wキー
if (this._selectedUnit && Input.isTriggered("pagedown")) { // Wキー
    const u = this._selectedUnit;
    u.acted = true;
    this._selectedUnit = null;
    this._moveRange = [];

    const newTeam = this._battle.endTurnIfNeeded();
    if (newTeam) {
        this.startPhaseBanner(newTeam);
    }
    this.redrawBoard();
    return;
}
  // ★ 追加：Qキーでターン終了
  if (Input.isTriggered("pageup")) { // Qキー
    this.forceEndTurn();
    return;
  }


    if (Input.isTriggered("cancel")) {
        this._selectedUnit = null;
        this._moveRange = []; // ★範囲消す
        return;
    }
const newTeam = this._battle.endTurnIfNeeded();
if (newTeam) {
    this.startPhaseBanner(newTeam);
}
    if (Input.isTriggered("ok")) {
        if (!this._selectedUnit) {
            this.trySelectUnit();
        } else {
            this.tryActSelectedUnit(); // ★移動 or 攻撃をまとめて処理
        }
    }
};

  if (moved) {
    this.updateCursorPosition();
  }
};




// --- ユニット検索ヘルパ ---
Scene_Tactics.prototype.unitAt = function(x, y) {
    const all = [...this._battle.allies, ...this._battle.enemies];
    return all.find(u => u.alive() && u.x === x && u.y === y) || null;
};

Scene_Tactics.prototype.isTileEmpty = function(x, y) {
    return !this.unitAt(x, y);
};

// --- ユニット選択 ---
Scene_Tactics.prototype.trySelectUnit = function() {
    const u = this.unitAt(this._cursorX, this._cursorY);
    if (!u) return;
    if (u.team !== this._battle.turnTeam) return;
    if (u.acted) return;
    this._selectedUnit = u;
     this._moveRange = this.computeMoveRange(u); // ★ここで使う
      this.redrawBoard();
};


Scene_Tactics.prototype.computeMoveRange = function(unit) {
  const w = this._battle.map.width;
  const h = this._battle.map.height;
  const maxStep = unit.mov;

  // "x,y" → コスト のマップ
  const costMap = {};
  const key = (x, y) => `${x},${y}`;

  const result = [];
  const queue = [];

  // 開始地点
  queue.push({ x: unit.x, y: unit.y, c: 0 });
  costMap[key(unit.x, unit.y)] = 0;

  while (queue.length > 0) {
    const node = queue.shift();
    const x = node.x;
    const y = node.y;
    const c = node.c;

    // 自分が立っているマスは「範囲」には入れない（ハイライトだけ）
    if (!(x === unit.x && y === unit.y)) {
      result.push({ x, y });
    }

    if (c >= maxStep) continue;

    const dirs = [
      { dx: 1, dy: 0 },
      { dx: -1, dy: 0 },
      { dx: 0, dy: 1 },
      { dx: 0, dy: -1 },
    ];

    for (const d of dirs) {
      const nx = x + d.dx;
      const ny = y + d.dy;
      if (!this._battle.map.inBounds(nx, ny)) continue;

      // 他ユニットがいるマスには入れない（※開始マスは例外）
      const other = this.unitAt(nx, ny);
      if (other && !(nx === unit.x && ny === unit.y)) continue;

      const nKey = key(nx, ny);
      const nCost = c + 1;

      if (costMap[nKey] == null || nCost < costMap[nKey]) {
        costMap[nKey] = nCost;
        queue.push({ x: nx, y: ny, c: nCost });
      }
    }
  }

  return result;
};


Scene_Tactics.prototype.forceEndTurn = function() {
  // 今のチーム
  const team = this._battle.turnTeam;

  // 現在生きてるユニット全員を「行動済み」にする
  const units = this._battle.livingUnits(team);
  for (const u of units) {
    u.acted = true;
  }

  // 選択状態・移動範囲をクリア
  this._selectedUnit = null;
  this._moveRange = [];

  // endTurnIfNeeded でターン切り替え（さっき拡張したやつ）
  const newTeam = this._battle.endTurnIfNeeded();
  if (newTeam) {
    // フェーズ演出があるならここで呼ぶ
    this.startPhaseBanner(newTeam);
  }

  // 画面更新
  this.redrawBoard();
};

// --- ユニット移動 ---
Scene_Tactics.prototype.tryMoveSelectedUnit = function() {
    const u = this._selectedUnit;
    if (!u) return;

    const tx = this._cursorX;
    const ty = this._cursorY;

    if (u.x === tx && u.y === ty) {
        this._selectedUnit = null;
        return;
    }

    const dist = Math.abs(tx - u.x) + Math.abs(ty - u.y);
    if (dist > u.mov) return;
    if (!this._battle.map.inBounds(tx, ty)) return;
    if (!this.isTileEmpty(tx, ty)) return;

    u.x = tx;
    u.y = ty;
    u.acted = true;
    this._selectedUnit = null;
    this._moveRange = []; // ★範囲消す

const newTeam = this._battle.endTurnIfNeeded();
if (newTeam) {
    this.startPhaseBanner(newTeam);
}

    this.redrawBoard();
};
Scene_Tactics.prototype.createCursorSprite = function() {
  const cellW = this._cellW;
  const cellH = this._cellH;
  const bmp = new Bitmap(cellW, cellH);

  // 黄色い枠っぽいのを描く
  const borderColor = "#ffff00";
  // 上
  bmp.fillRect(0, 0, cellW, 2, borderColor);
  // 下
  bmp.fillRect(0, cellH - 2, cellW, 2, borderColor);
  // 左
  bmp.fillRect(0, 0, 2, cellH, borderColor);
  // 右
  bmp.fillRect(cellW - 2, 0, 2, cellH, borderColor);

  this._cursorSprite = new Sprite(bmp);
  this.addChild(this._cursorSprite);


  this.createStatusWindow();
};
Scene_Tactics.prototype.refreshStatusWindow = function() {
    if (!this._statusWindow) return;

    const win = this._statusWindow;
    win.contents.clear();

    // ターン表示
    const turnText = (this._battle.turnTeam === 'ally') ? "味方ターン" : "敵ターン";
    const turnY = -6;
    win.drawText(turnText, 0, turnY, win.contentsWidth(), 'left');

    // 選択中ユニットがいるなら、そのステも表示
    const u = this._selectedUnit;
    if (u) {
        const y = 20;
        const text =
            `${u.name}  HP:${u.hp}/${u.maxHp}  ` +
            `ATK:${u.atk}  DEF:${u.def}  MOV:${u.mov}`;
        win.drawText(text, 0, y, win.contentsWidth(), 'left');
    }
};
Scene_Tactics.prototype.createStatusWindow = function() {
    const h = 72;
    const yOffset = 0; // ← 上に持ち上げたいピクセル数（お好みで）
    const rect = new Rectangle(
        0,
        Graphics.boxHeight - h- yOffset, // ← ここをちょっと引く
        Graphics.boxWidth,
        h
    );
    this._statusWindow = new Window_Base(rect);
    this.addChild(this._statusWindow);
    this.refreshStatusWindow();
};



Scene_Tactics.prototype.updateCursorPosition = function() {
  this._cursorSprite.x = this._cursorX * this._cellW;
  this._cursorSprite.y = this._cursorY * this._cellH;
};



